<?php
require_once '../config/config.php';
requireLogin();

$pdo = getDBConnection();
$message = '';
$error = '';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'toggle_status') {
        $channelId = intval($_POST['channel_id']);
        $newStatus = intval($_POST['new_status']);
        
        try {
            $stmt = $pdo->prepare("UPDATE tv_channels SET is_active = ? WHERE id = ?");
            $stmt->execute([$newStatus, $channelId]);
            $message = "Channel status updated successfully.";
        } catch (Exception $e) {
            $error = "Error updating channel status: " . $e->getMessage();
        }
    }
    
    elseif ($action === 'delete_channel') {
        $channelId = intval($_POST['channel_id']);
        
        try {
            $stmt = $pdo->prepare("DELETE FROM tv_channels WHERE id = ?");
            $stmt->execute([$channelId]);
            $message = "Channel deleted successfully.";
        } catch (Exception $e) {
            $error = "Error deleting channel: " . $e->getMessage();
        }
    }
    
    elseif ($action === 'toggle_pin') {
        $channelId = intval($_POST['channel_id']);
        $newPinStatus = intval($_POST['new_pin_status']);
        
        try {
            $stmt = $pdo->prepare("UPDATE tv_channels SET is_pinned = ? WHERE id = ?");
            $stmt->execute([$newPinStatus, $channelId]);
            $message = "Channel pin status updated successfully.";
        } catch (Exception $e) {
            $error = "Error updating channel pin status: " . $e->getMessage();
        }
    }
    
    elseif ($action === 'update_channel') {
        $channelId = intval($_POST['channel_id']);
        $name = sanitizeInput($_POST['name']);
        $streamUrl = sanitizeInput($_POST['stream_url']);
        $category = sanitizeInput($_POST['category']);
        $logoUrl = sanitizeInput($_POST['logo_url']);
        $logoPath = '';
        
        // Handle logo upload
        if (isset($_FILES['logo_file']) && $_FILES['logo_file']['error'] === UPLOAD_ERR_OK) {
            $logoUpload = uploadFile($_FILES['logo_file'], LOGO_UPLOAD_PATH, ['jpg', 'jpeg', 'png', 'gif', 'webp']);
            if ($logoUpload['success']) {
                $logoPath = $logoUpload['file_path'];
            }
        }
        
        // Use logo URL if no file uploaded
        if (empty($logoPath) && !empty($logoUrl)) {
            $logoPath = $logoUrl;
        }
        
        try {
            if (!empty($logoPath)) {
                $stmt = $pdo->prepare("UPDATE tv_channels SET name = ?, stream_url = ?, category = ?, logo_url = ? WHERE id = ?");
                $stmt->execute([$name, $streamUrl, $category, $logoPath, $channelId]);
            } else {
                $stmt = $pdo->prepare("UPDATE tv_channels SET name = ?, stream_url = ?, category = ? WHERE id = ?");
                $stmt->execute([$name, $streamUrl, $category, $channelId]);
            }
            $message = "Channel updated successfully.";
        } catch (Exception $e) {
            $error = "Error updating channel: " . $e->getMessage();
        }
    }
}

// Get all channels (pinned first, then by created date)
try {
    $channels = $pdo->query("SELECT * FROM tv_channels ORDER BY is_pinned DESC, created_at DESC")->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    // If is_pinned column doesn't exist, fallback to created_at order
    $channels = $pdo->query("SELECT * FROM tv_channels ORDER BY created_at DESC")->fetchAll(PDO::FETCH_ASSOC);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Channels - Live TV Admin</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .sidebar {
            background: linear-gradient(135deg, #8B0000 0%, #DC143C 100%);
            min-height: 100vh;
            color: white;
        }
        .sidebar .nav-link {
            color: rgba(255, 255, 255, 0.8);
            padding: 12px 20px;
            border-radius: 8px;
            margin: 5px 10px;
            transition: all 0.3s;
        }
        .sidebar .nav-link:hover, .sidebar .nav-link.active {
            color: white;
            background: rgba(255, 255, 255, 0.2);
            transform: translateX(5px);
        }
        .main-content {
            background: #000000;
            min-height: 100vh;
            color: white;
        }
        
        .main-content h1, .main-content h2, .main-content h3, .main-content h4, .main-content h5, .main-content h6 {
            color: white !important;
        }
        
        .main-content p, .main-content span, .main-content div {
            color: white !important;
        }
        
        .main-content .card-header {
            background: rgba(220, 20, 60, 0.2);
            color: white;
            border-bottom: 1px solid rgba(220, 20, 60, 0.3);
        }
        
        .main-content .card-body {
            background: rgba(255, 255, 255, 0.05);
            color: white;
        }
        
        .main-content .form-label {
            color: white !important;
        }
        
        .main-content .form-control {
            background: rgba(255, 255, 255, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.3);
            color: white;
        }
        
        .main-content .form-control::placeholder {
            color: rgba(255, 255, 255, 0.7);
        }
        
        .main-content .form-select {
            background: rgba(255, 255, 255, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.3);
            color: white;
        }
        
        .main-content .table {
            color: white;
        }
        
        .main-content .table th {
            color: white;
            border-color: rgba(255, 255, 255, 0.2);
        }
        
        .main-content .table td {
            color: white;
            border-color: rgba(255, 255, 255, 0.1);
        }
        
        .main-content .badge {
            color: white;
        }
        
        .main-content .btn-outline-primary {
            color: #DC143C;
            border-color: #DC143C;
        }
        
        .main-content .btn-outline-primary:hover {
            background-color: #DC143C;
            border-color: #DC143C;
            color: white;
        }
        
        .main-content .btn-outline-warning {
            color: #ffc107;
            border-color: #ffc107;
        }
        
        .main-content .btn-outline-warning:hover {
            background-color: #ffc107;
            border-color: #ffc107;
            color: black;
        }
        
        .main-content .btn-outline-success {
            color: #198754;
            border-color: #198754;
        }
        
        .main-content .btn-outline-success:hover {
            background-color: #198754;
            border-color: #198754;
            color: white;
        }
        
        .main-content .btn-outline-danger {
            color: #dc3545;
            border-color: #dc3545;
        }
        
        .main-content .btn-outline-danger:hover {
            background-color: #dc3545;
            border-color: #dc3545;
            color: white;
        }
        .card {
            border: none;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.3);
            background: rgba(255, 255, 255, 0.1);
            backdrop-filter: blur(10px);
        }
        .btn-primary {
            background: linear-gradient(135deg, #8B0000 0%, #DC143C 100%);
            border: none;
            border-radius: 8px;
            padding: 10px 20px;
            font-weight: 600;
        }
        .channel-logo {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            object-fit: cover;
        }
        .status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.8rem;
        }
        .modal-content {
            border-radius: 15px;
            border: none;
        }
        .modal-header {
            background: linear-gradient(135deg, #8B0000 0%, #DC143C 100%);
            color: white;
            border-radius: 15px 15px 0 0;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 px-0">
                <div class="sidebar">
                    <div class="p-3">
                        <h4><i class="fas fa-tv me-2"></i>Live TV Admin</h4>
                        <p class="text-light">Welcome, <?php echo $_SESSION['admin_username']; ?></p>
                    </div>
                    <nav class="nav flex-column">
                        <a class="nav-link" href="dashboard.php">
                            <i class="fas fa-tachometer-alt me-2"></i>Dashboard
                        </a>
                        <a class="nav-link active" href="channels.php">
                            <i class="fas fa-tv me-2"></i>Manage Channels
                        </a>
                        <a class="nav-link" href="upload-m3u.php">
                            <i class="fas fa-upload me-2"></i>Upload M3U
                        </a>
                        <a class="nav-link" href="playlists.php">
                            <i class="fas fa-list me-2"></i>M3U Playlists
                        </a>
                        <a class="nav-link" href="../index.php" target="_blank">
                            <i class="fas fa-external-link-alt me-2"></i>View Website
                        </a>
                        <a class="nav-link" href="logout.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Logout
                        </a>
                    </nav>
                </div>
            </div>
            
            <!-- Main Content -->
            <div class="col-md-9 col-lg-10">
                <div class="main-content p-4">
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <h2><i class="fas fa-tv me-2"></i>Manage Channels</h2>
                        <div>
                            <a href="upload-m3u.php" class="btn btn-primary">
                                <i class="fas fa-plus me-2"></i>Add New Channel
                            </a>
                        </div>
                    </div>
                    
                    <?php if ($message): ?>
                        <div class="alert alert-success alert-dismissible fade show" role="alert">
                            <i class="fas fa-check-circle me-2"></i><?php echo $message; ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>
                    
                    <?php if ($error): ?>
                        <div class="alert alert-danger alert-dismissible fade show" role="alert">
                            <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>
                    
                    <?php if (empty($channels)): ?>
                        <div class="card">
                            <div class="card-body text-center py-5">
                                <i class="fas fa-tv fa-4x text-muted mb-3"></i>
                                <h4 class="text-muted">No Channels Found</h4>
                                <p class="text-muted">Start by adding your first TV channel.</p>
                                <a href="upload-m3u.php" class="btn btn-primary">
                                    <i class="fas fa-plus me-2"></i>Add Your First Channel
                                </a>
                            </div>
                        </div>
                    <?php else: ?>
                        <div class="card">
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead>
                                            <tr>
                                                <th>Logo</th>
                                                <th>Channel Name</th>
                                                <th>Stream URL</th>
                                                <th>Category</th>
                                                <th>Status</th>
                                                <th>Created</th>
                                                <th>Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($channels as $channel): ?>
                                                <tr>
                                                    <td>
                                                        <?php if ($channel['logo_url']): ?>
                                                            <img src="../<?php echo $channel['logo_url']; ?>" 
                                                                 alt="<?php echo htmlspecialchars($channel['name']); ?>" 
                                                                 class="channel-logo">
                                                        <?php else: ?>
                                                            <div class="channel-logo bg-secondary d-flex align-items-center justify-content-center">
                                                                <i class="fas fa-tv text-white"></i>
                                                            </div>
                                                        <?php endif; ?>
                                                    </td>
                                                    <td>
                                                        <strong><?php echo htmlspecialchars($channel['name']); ?></strong>
                                                    </td>
                                                    <td>
                                                        <small class="text-muted">
                                                            <?php echo htmlspecialchars(substr($channel['stream_url'], 0, 50) . '...'); ?>
                                                        </small>
                                                    </td>
                                                    <td>
                                                        <span class="badge bg-info">
                                                            <?php echo htmlspecialchars($channel['category'] ?? 'General'); ?>
                                                        </span>
                                                    </td>
                                                    <td>
                                                        <?php if ($channel['is_active']): ?>
                                                            <span class="badge bg-success">Active</span>
                                                        <?php else: ?>
                                                            <span class="badge bg-danger">Inactive</span>
                                                        <?php endif; ?>
                                                    </td>
                                                    <td>
                                                        <small class="text-muted">
                                                            <?php echo date('M j, Y', strtotime($channel['created_at'])); ?>
                                                        </small>
                                                    </td>
                                                    <td>
                                                        <div class="btn-group" role="group">
                                                            <button class="btn btn-sm btn-outline-primary" 
                                                                    onclick="editChannel(<?php echo htmlspecialchars(json_encode($channel)); ?>)">
                                                                <i class="fas fa-edit"></i>
                                                            </button>
                                                            
                                                            <?php if (isset($channel['is_pinned'])): ?>
                                                            <form method="POST" style="display: inline;">
                                                                <input type="hidden" name="action" value="toggle_pin">
                                                                <input type="hidden" name="channel_id" value="<?php echo $channel['id']; ?>">
                                                                <input type="hidden" name="new_pin_status" value="<?php echo $channel['is_pinned'] ? 0 : 1; ?>">
                                                                <button type="submit" class="btn btn-sm btn-outline-<?php echo $channel['is_pinned'] ? 'warning' : 'info'; ?>"
                                                                        title="<?php echo $channel['is_pinned'] ? 'Unpin from top' : 'Pin to top'; ?>">
                                                                    <i class="fas fa-thumbtack"></i>
                                                                </button>
                                                            </form>
                                                            <?php endif; ?>
                                                            
                                                            <form method="POST" style="display: inline;">
                                                                <input type="hidden" name="action" value="toggle_status">
                                                                <input type="hidden" name="channel_id" value="<?php echo $channel['id']; ?>">
                                                                <input type="hidden" name="new_status" value="<?php echo $channel['is_active'] ? 0 : 1; ?>">
                                                                <button type="submit" class="btn btn-sm btn-outline-<?php echo $channel['is_active'] ? 'warning' : 'success'; ?>"
                                                                        onclick="return confirm('Are you sure you want to <?php echo $channel['is_active'] ? 'deactivate' : 'activate'; ?> this channel?')">
                                                                    <i class="fas fa-<?php echo $channel['is_active'] ? 'pause' : 'play'; ?>"></i>
                                                                </button>
                                                            </form>
                                                            
                                                            <form method="POST" style="display: inline;">
                                                                <input type="hidden" name="action" value="delete_channel">
                                                                <input type="hidden" name="channel_id" value="<?php echo $channel['id']; ?>">
                                                                <button type="submit" class="btn btn-sm btn-outline-danger"
                                                                        onclick="return confirm('Are you sure you want to delete this channel? This action cannot be undone.')">
                                                                    <i class="fas fa-trash"></i>
                                                                </button>
                                                            </form>
                                                        </div>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Edit Channel Modal -->
    <div class="modal fade" id="editChannelModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-edit me-2"></i>Edit Channel
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST" id="editChannelForm" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="update_channel">
                    <input type="hidden" name="channel_id" id="editChannelId">
                    
                    <div class="modal-body">
                        <div class="mb-3">
                            <label for="editName" class="form-label">Channel Name</label>
                            <input type="text" class="form-control" id="editName" name="name" required>
                        </div>
                        
                        <div class="mb-3">
                            <label for="editStreamUrl" class="form-label">Stream URL</label>
                            <input type="url" class="form-control" id="editStreamUrl" name="stream_url" required>
                        </div>
                        
                        <div class="mb-3">
                            <label for="editCategory" class="form-label">Category</label>
                            <select class="form-select" id="editCategory" name="category">
                                <option value="General">General</option>
                                <option value="News">News</option>
                                <option value="Sports">Sports</option>
                                <option value="Movies">Movies</option>
                                <option value="Kids">Kids</option>
                                <option value="Music">Music</option>
                                <option value="Documentary">Documentary</option>
                            </select>
                        </div>
                        
                        <div class="mb-3">
                            <label for="editLogoUrl" class="form-label">Logo URL</label>
                            <input type="url" class="form-control" id="editLogoUrl" name="logo_url" 
                                   placeholder="https://example.com/logo.png">
                        </div>
                        
                        <div class="mb-3">
                            <label for="editLogoFile" class="form-label">Or Upload New Logo</label>
                            <input type="file" class="form-control" id="editLogoFile" name="logo_file" 
                                   accept="image/*">
                        </div>
                    </div>
                    
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i>Update Channel
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function editChannel(channel) {
            document.getElementById('editChannelId').value = channel.id;
            document.getElementById('editName').value = channel.name;
            document.getElementById('editStreamUrl').value = channel.stream_url;
            document.getElementById('editCategory').value = channel.category || 'General';
            document.getElementById('editLogoUrl').value = channel.logo_url || '';
            
            const modal = new bootstrap.Modal(document.getElementById('editChannelModal'));
            modal.show();
        }
    </script>
</body>
</html>
