<?php
// M3U Playlist Parser
class M3UParser {
    
    public static function parseM3U($m3uContent) {
        $channels = [];
        $lines = explode("\n", $m3uContent);
        $currentChannel = null;
        
        foreach ($lines as $line) {
            $line = trim($line);
            
            // Skip empty lines and comments
            if (empty($line) || strpos($line, '#') === 0) {
                continue;
            }
            
            // Check if this is a stream URL
            if (self::isStreamUrl($line)) {
                if ($currentChannel) {
                    $currentChannel['stream_url'] = $line;
                    $channels[] = $currentChannel;
                    $currentChannel = null;
                }
            } else {
                // This might be channel info
                $currentChannel = [
                    'name' => $line,
                    'stream_url' => '',
                    'logo_url' => '',
                    'category' => 'General'
                ];
            }
        }
        
        return $channels;
    }
    
    public static function parseM3UFromFile($filePath) {
        if (!file_exists($filePath)) {
            return false;
        }
        
        $content = file_get_contents($filePath);
        return self::parseM3U($content);
    }
    
    public static function parseM3UFromUrl($url) {
        $content = @file_get_contents($url);
        if ($content === false) {
            return false;
        }
        
        return self::parseM3U($content);
    }
    
    private static function isStreamUrl($line) {
        $streamProtocols = ['http://', 'https://', 'rtmp://', 'rtsp://', 'udp://'];
        
        foreach ($streamProtocols as $protocol) {
            if (strpos($line, $protocol) === 0) {
                return true;
            }
        }
        
        return false;
    }
    
    public static function extractChannelInfo($m3uContent) {
        $channels = [];
        $lines = explode("\n", $m3uContent);
        $currentChannel = null;
        
        foreach ($lines as $line) {
            $line = trim($line);
            
            if (empty($line)) {
                continue;
            }
            
            // Parse EXTINF line for channel info
            if (strpos($line, '#EXTINF:') === 0) {
                $currentChannel = self::parseExtInf($line);
            }
            // Check if this is a stream URL
            elseif (self::isStreamUrl($line) && $currentChannel) {
                $currentChannel['stream_url'] = $line;
                $channels[] = $currentChannel;
                $currentChannel = null;
            }
        }
        
        return $channels;
    }
    
    private static function parseExtInf($extInfLine) {
        // Remove #EXTINF: prefix
        $extInfLine = substr($extInfLine, 8);
        
        // Split by comma to separate duration and info
        $parts = explode(',', $extInfLine, 2);
        $duration = isset($parts[0]) ? intval($parts[0]) : -1;
        $info = isset($parts[1]) ? $parts[1] : '';
        
        $channel = [
            'name' => $info,
            'duration' => $duration,
            'stream_url' => '',
            'logo_url' => '',
            'category' => 'General'
        ];
        
        // Try to extract logo URL from tvg-logo attribute
        if (preg_match('/tvg-logo="([^"]+)"/', $extInfLine, $matches)) {
            $channel['logo_url'] = $matches[1];
        }
        
        // Try to extract channel name from tvg-name attribute
        if (preg_match('/tvg-name="([^"]+)"/', $extInfLine, $matches)) {
            $channel['name'] = $matches[1];
        }
        
        // Try to extract group title (category)
        if (preg_match('/group-title="([^"]+)"/', $extInfLine, $matches)) {
            $channel['category'] = $matches[1];
        }
        
        return $channel;
    }
}
?>
