<?php
// Live TV Setup Script
require_once 'config/config.php';

$setupComplete = false;
$errors = [];

// Check if database connection works
try {
    $pdo = getDBConnection();
    $setupComplete = true;
} catch (Exception $e) {
    $errors[] = "Database connection failed: " . $e->getMessage();
}

// Create necessary directories
$directories = [
    'uploads',
    'uploads/m3u',
    'uploads/logos',
    'config',
    'admin',
    'includes'
];

foreach ($directories as $dir) {
    if (!file_exists($dir)) {
        if (!mkdir($dir, 0777, true)) {
            $errors[] = "Failed to create directory: $dir";
        }
    }
}

// Create .htaccess for security
$htaccessContent = "Options -Indexes\nDeny from all";
file_put_contents('uploads/.htaccess', $htaccessContent);

// Check if database tables exist and create admin user
if ($setupComplete) {
    try {
        $tables = $pdo->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
        $requiredTables = ['admin_users', 'tv_channels', 'm3u_playlists'];
        
        foreach ($requiredTables as $table) {
            if (!in_array($table, $tables)) {
                $errors[] = "Required table '$table' does not exist. Please run the database.sql file.";
            }
        }
        
        // Create admin user if tables exist
        if (empty($errors)) {
            $stmt = $pdo->prepare("SELECT id FROM admin_users WHERE username = ?");
            $stmt->execute(['admin']);
            $existingUser = $stmt->fetch();
            
            if (!$existingUser) {
                $hashedPassword = password_hash('admin123', PASSWORD_DEFAULT);
                $stmt = $pdo->prepare("INSERT INTO admin_users (username, password, email) VALUES (?, ?, ?)");
                $stmt->execute(['admin', $hashedPassword, 'admin@example.com']);
            }
        }
    } catch (Exception $e) {
        $errors[] = "Database error: " . $e->getMessage();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Live TV Setup</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
        }
        .setup-card {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 15px;
            box-shadow: 0 15px 35px rgba(0, 0, 0, 0.1);
            backdrop-filter: blur(10px);
        }
        .setup-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 15px 15px 0 0;
            padding: 2rem;
            text-align: center;
        }
        .status-item {
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1rem;
            display: flex;
            align-items: center;
        }
        .status-success {
            background: rgba(40, 167, 69, 0.1);
            border-left: 4px solid #28a745;
        }
        .status-error {
            background: rgba(220, 53, 69, 0.1);
            border-left: 4px solid #dc3545;
        }
        .btn-setup {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border: none;
            padding: 12px 30px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 1px;
            border-radius: 8px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="row justify-content-center">
            <div class="col-md-8 col-lg-6">
                <div class="card setup-card">
                    <div class="setup-header">
                        <i class="fas fa-tv fa-3x mb-3"></i>
                        <h3>Live TV Setup</h3>
                        <p class="mb-0">System Configuration Check</p>
                    </div>
                    <div class="card-body p-4">
                        <?php if (empty($errors)): ?>
                            <div class="status-item status-success">
                                <i class="fas fa-check-circle text-success fa-2x me-3"></i>
                                <div>
                                    <h5 class="mb-1">Setup Complete!</h5>
                                    <p class="mb-0">Your Live TV system is ready to use.</p>
                                </div>
                            </div>
                            
                            <div class="text-center mt-4">
                                <a href="index.php" class="btn btn-primary btn-setup me-3">
                                    <i class="fas fa-tv me-2"></i>View Website
                                </a>
                                <a href="admin/login.php" class="btn btn-outline-primary btn-setup">
                                    <i class="fas fa-cog me-2"></i>Admin Panel
                                </a>
                            </div>
                            
                            <div class="mt-4">
                                <h6><i class="fas fa-info-circle me-2"></i>Default Admin Credentials:</h6>
                                <div class="bg-light p-3 rounded">
                                    <strong>Username:</strong> admin<br>
                                    <strong>Password:</strong> admin123
                                </div>
                            </div>
                        <?php else: ?>
                            <div class="status-item status-error">
                                <i class="fas fa-exclamation-triangle text-danger fa-2x me-3"></i>
                                <div>
                                    <h5 class="mb-1">Setup Issues Found</h5>
                                    <p class="mb-0">Please resolve the following issues:</p>
                                </div>
                            </div>
                            
                            <?php foreach ($errors as $error): ?>
                                <div class="alert alert-danger">
                                    <i class="fas fa-times-circle me-2"></i><?php echo htmlspecialchars($error); ?>
                                </div>
                            <?php endforeach; ?>
                            
                            <div class="mt-4">
                                <h6><i class="fas fa-wrench me-2"></i>Setup Instructions:</h6>
                                <ol>
                                    <li>Make sure XAMPP is running</li>
                                    <li>Create a database named <code>live_tv_db</code> in phpMyAdmin</li>
                                    <li>Import the <code>database.sql</code> file</li>
                                    <li>Refresh this page</li>
                                </ol>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
